﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using System.Net.Sockets;
using System.Net;
using Communication.IO.Tools;
using System.Threading;
using System.Reflection;
using System.Net.NetworkInformation;
using System.IO;

namespace Sentis_ToF_M100_Visualizer
{
    public partial class SentisTofVisualizerForm : Form
    {

        public SentisTofVisualizerForm()
        {
            // preserve user settings from last build
            System.Reflection.Assembly a = System.Reflection.Assembly.GetExecutingAssembly();
            if (Properties.Settings.Default.ApplicationVersion != a.GetName().Version.ToString())
            {
                Properties.Settings.Default.Upgrade();
                Properties.Settings.Default.ApplicationVersion = a.GetName().Version.ToString();
            }
            InitializeComponent();
            tbTcpAddr.Text = Properties.Settings.Default.tcp_addr;
            tbTcpPort.Text = Properties.Settings.Default.tcp_port;
            txtStatsFrameCount_TextChanged(null, null);
            SentisTofVisualizerForm_Resize(null, null);
            udpStreamReceiverThread = new Thread(udpStreamReceiver);
            udpStreamReceiverThread.Name = "UDPstreamReceiver";
            udpStreamReceiverThread.Start();
            frameHandlerThread = new Thread(frameHandler);
            frameHandlerThread.Name = "FrameHandler";
            frameHandlerThread.Start();
            this.Text += " V" + Assembly.GetExecutingAssembly().GetName().Version.ToString(3);
        }


        #region ControlInterfaceTcp
        // Private members for control interface communication
        private const int CONTROL_IF_HEADER_SIZE = 64;
        private const byte PREAMBLE_0 = 0xa1;
        private const byte PREAMBLE_1 = 0xec;
        private const byte PROTOCOL_VERSION = 3;
        private const byte CMD_CODE_WRITE = 4;
        private const byte CMD_CODE_READ = 3;
        private const byte FLAGS_IGNORE_DATACRC3 = 1;
        private const UInt32 LENGTH_STD_CMD = 2;

        private const UInt16 REG_INTEGRATION_TIME = 5;
        private const UInt16 REG_FRAME_RATE = 10;


        // This struct holds the data from the control interface header
        private struct ControlIfHeader
        {
            public UInt16 preamble;
            public byte protocolVersion;
            public byte command;
            public byte subCommand;
            public byte status;
            public UInt16 flags;
            public UInt32 length;
            public byte headerData0, headerData1, headerData2, headerData3;
            //public byte reserved[];
            public UInt32 dataCrc32;
            public UInt16 headerCrc16;
        }


        // A function to read a register
        // It establishes a TCP connection, composes the stream, sends it and reads the response
        // Then the connection is closed again
        private byte readRegister(UInt16 regAddr, out UInt16 result)
        {
            lock (controlInterfaceLock)
            {
                result = 0;
                IPEndPoint cgwIP;
                try
                {
                    cgwIP = new IPEndPoint(IPAddress.Parse(tbTcpAddr.Text), Convert.ToInt32(tbTcpPort.Text));
                }
                catch (Exception exc)
                {
                    MessageBox.Show("Control interface: Enter a valid IP address and port, please");
                    return 255;
                }
                TcpClient configTcpClient = null;
                NetworkStream configTcpStream = null;
                try
                {
                    configTcpClient = new TcpClient();
                    configTcpClient.ReceiveTimeout = 1000;
                    configTcpClient.ReceiveTimeout = 1000;
                    configTcpClient.ReceiveBufferSize = 7683000;
                    configTcpClient.SendBufferSize = 7683000;
                    configTcpClient.NoDelay = true;
                    configTcpClient.Connect(cgwIP);
                    configTcpClient.Client.ReceiveTimeout = 1000;
                    configTcpClient.Client.SendTimeout = 1000;
                    configTcpClient.Client.ReceiveBufferSize = 7683000;
                    configTcpClient.Client.SendBufferSize = 7683000;
                    configTcpStream = configTcpClient.GetStream();
                    configTcpStream.ReadTimeout = 1000;
                    configTcpStream.WriteTimeout = 1000;
                    if (!configTcpClient.Connected)
                    {
                        MessageBox.Show("Unable to establish connection");
                        return 255;
                    }

                    // send command
                    List<byte> sendHeader = composeBuffer(false, regAddr, 0);
                    configTcpStream.Write(sendHeader.ToArray(), 0, sendHeader.Count);
                    // done sending command

                    // read response
                    byte[] readBuffer = new byte[CONTROL_IF_HEADER_SIZE];
                    readBuffer[0] = (byte)configTcpStream.ReadByte();
                    readBuffer[1] = (byte)configTcpStream.ReadByte();
                    while (configTcpClient.Connected && (readBuffer[0] != PREAMBLE_0) && (readBuffer[1] != PREAMBLE_1))
                    {
                        readBuffer[0] = readBuffer[1];
                        readBuffer[1] = (byte)configTcpStream.ReadByte();
                    }

                    int bytesRead = 2;
                    readBuffer[bytesRead++] = (byte)configTcpStream.ReadByte();
                    while (bytesRead < CONTROL_IF_HEADER_SIZE)
                    {
                        bytesRead += configTcpStream.Read(readBuffer, bytesRead, CONTROL_IF_HEADER_SIZE - bytesRead);
                    }
                    // done reading response

                    byte[] data = new byte[0];
                    ControlIfHeader responseHeader = bufferToHeader(readBuffer);
                    // calculate headerCrc16
                    CRCTool crcTool16 = new CRCTool();
                    crcTool16.Init(CRCTool.CRCCode.CRC16);
                    UInt16 headerCrc16 = (UInt16)crcTool16.crctable16(readBuffer.ToArray(), 2, 60);
                    if (responseHeader.headerCrc16 != headerCrc16)
                    {
                        configTcpStream.Close();
                        configTcpClient.Close();
                        return 255;
                    }

                    if (responseHeader.length != LENGTH_STD_CMD)
                    {
                        configTcpStream.Close();
                        configTcpClient.Close();
                        return 255;
                    }
                    // continue reading stream (data)
                    data = new byte[LENGTH_STD_CMD];
                    bytesRead = 0;
                    while (bytesRead < LENGTH_STD_CMD)
                    {
                        bytesRead += configTcpStream.Read(data, bytesRead, (int)(LENGTH_STD_CMD - bytesRead));
                    }
                    // done reading data
                    configTcpStream.Close();
                    configTcpClient.Close();
                    result = (UInt16)((data[0] << 8) | data[1]);
                    return responseHeader.status;
                }
                catch (IOException exc)
                {
                    configTcpStream.Close();
                    configTcpClient.Close();
                    return 255;
                }
                catch (SocketException exc)
                {
                    return 255;
                }
            }
        }

        // A function to write a register
        // It establishes a TCP connection, composes the stream, sends it and reads the response
        // Then the connection is closed again
        private byte writeRegister(UInt16 regAddr, UInt16 valueToWrite)
        {
            lock (controlInterfaceLock)
            {
                IPEndPoint cgwIP;
                try
                {
                    cgwIP = new IPEndPoint(IPAddress.Parse(tbTcpAddr.Text), Convert.ToInt32(tbTcpPort.Text));
                }
                catch (Exception exc)
                {
                    MessageBox.Show("Control interface: Enter a valid IP address and port, please");
                    return 255;
                }
                TcpClient configTcpClient = null;
                NetworkStream configTcpStream = null;
                try
                {
                    configTcpClient = new TcpClient();
                    configTcpClient.ReceiveTimeout = 1000;
                    configTcpClient.ReceiveTimeout = 1000;
                    configTcpClient.ReceiveBufferSize = 7683000;
                    configTcpClient.SendBufferSize = 7683000;
                    configTcpClient.NoDelay = true;
                    configTcpClient.Connect(cgwIP);
                    configTcpClient.Client.ReceiveTimeout = 1000;
                    configTcpClient.Client.SendTimeout = 1000;
                    configTcpClient.Client.ReceiveBufferSize = 7683000;
                    configTcpClient.Client.SendBufferSize = 7683000;
                    configTcpStream = configTcpClient.GetStream();
                    configTcpStream.ReadTimeout = 1000;
                    configTcpStream.WriteTimeout = 1000;
                    if (!configTcpClient.Connected)
                    {
                        MessageBox.Show("Unable to establish connection");
                        return 255;
                    }

                    // send command
                    List<byte> sendHeader = composeBuffer(true, regAddr, valueToWrite);
                    configTcpStream.Write(sendHeader.ToArray(), 0, sendHeader.Count);
                    // done sending command

                    // read response
                    byte[] readBuffer = new byte[CONTROL_IF_HEADER_SIZE];
                    readBuffer[0] = (byte)configTcpStream.ReadByte();
                    readBuffer[1] = (byte)configTcpStream.ReadByte();
                    while (configTcpClient.Connected && (readBuffer[0] != PREAMBLE_0) && (readBuffer[1] != PREAMBLE_1))
                    {
                        readBuffer[0] = readBuffer[1];
                        readBuffer[1] = (byte)configTcpStream.ReadByte();
                    }

                    int bytesRead = 2;
                    readBuffer[bytesRead++] = (byte)configTcpStream.ReadByte();
                    while (bytesRead < CONTROL_IF_HEADER_SIZE)
                    {
                        bytesRead += configTcpStream.Read(readBuffer, bytesRead, CONTROL_IF_HEADER_SIZE - bytesRead);
                    }
                    // done reading response

                    configTcpStream.Close();
                    configTcpClient.Close();

                    ControlIfHeader responseHeader = bufferToHeader(readBuffer);
                    // calculate headerCrc16
                    CRCTool crcTool16 = new CRCTool();
                    crcTool16.Init(CRCTool.CRCCode.CRC16);
                    UInt16 headerCrc16 = (UInt16)crcTool16.crctable16(readBuffer.ToArray(), 2, 60);
                    if (responseHeader.headerCrc16 != headerCrc16)
                    {
                        return 255;
                    }
                    return responseHeader.status;
                }
                catch (IOException exc)
                {
                    configTcpStream.Close();
                    configTcpClient.Close();
                    return 255;
                }
                catch (SocketException exc)
                {
                    return 255;
                }
            }
        }


        // Function to compose the header plus data stream meeting the control interface protocol
        private List<byte> composeBuffer(bool write, UInt16 regAddr, UInt16 regValue)
        {
            List<byte> result = new List<byte>();
            // preamble
            result.Add(PREAMBLE_0);
            result.Add(PREAMBLE_1);
            // protocol version
            result.Add(PROTOCOL_VERSION);
            // command code
            if (write)
            {
                result.Add(CMD_CODE_WRITE);
            }
            else
            {
                result.Add(CMD_CODE_READ);
            }
            // sub command
            result.Add(0);
            // status
            result.Add(0);
            // flags
            result.Add((byte) (FLAGS_IGNORE_DATACRC3 >> 8));
            result.Add((byte) FLAGS_IGNORE_DATACRC3);
            // data length
            result.Add((byte) (LENGTH_STD_CMD >> 24));
            result.Add((byte) (LENGTH_STD_CMD >> 16));
            result.Add((byte) (LENGTH_STD_CMD >> 8));
            result.Add((byte) LENGTH_STD_CMD);
            // header data
            result.Add((byte) (regAddr >> 8));
            result.Add((byte) regAddr);
            result.Add(0);
            result.Add(0);
            // 42 reserved bytes
            for (int i = 0; i < 42; i ++) {
                result.Add(0);
            }
            // data CRC checksum (not used)
            result.Add(0);
            result.Add(0);
            result.Add(0);
            result.Add(0);

            // calculate headerCrc16
            CRCTool crcTool16 = new CRCTool();
            crcTool16.Init(CRCTool.CRCCode.CRC16);
            UInt16 headerCrc16 = (UInt16)crcTool16.crctable16(result.ToArray(), 2, 60);

            result.Add((byte)(headerCrc16 >> 8));
            result.Add((byte)(headerCrc16));
            if (write)
            {
                result.Add((byte)(regValue >> 8));
                result.Add((byte)regValue);
            }

            return result;
        }


        // Function to fill the control interface header struct using a header plus data stream
        private ControlIfHeader bufferToHeader(byte[] buffer)
        {
            int bufferIndex = 0;
            ControlIfHeader header = new ControlIfHeader();
            header.preamble = (UInt16)(((UInt16)buffer[bufferIndex] << 8) | (UInt16)buffer[bufferIndex + 1]);
            bufferIndex += 2;
            header.protocolVersion = buffer[bufferIndex++];
            header.command = buffer[bufferIndex++];
            header.subCommand = buffer[bufferIndex++];
            header.status = buffer[bufferIndex++];
            header.flags = (UInt16)(((UInt16)buffer[bufferIndex] << 8) | (UInt16)buffer[bufferIndex + 1]);
            bufferIndex += 2;
            header.length = ((UInt32)buffer[bufferIndex] << 24) | ((UInt32)buffer[bufferIndex + 1] << 16) | ((UInt32)buffer[bufferIndex + 2] << 8) | buffer[bufferIndex + 3];
            bufferIndex += 4;
            header.headerData0 = buffer[bufferIndex++];
            header.headerData1 = buffer[bufferIndex++];
            header.headerData2 = buffer[bufferIndex++];
            header.headerData3 = buffer[bufferIndex++];
            // 42 bytes reserved
            bufferIndex += 42;
            header.dataCrc32 = ((UInt32)buffer[bufferIndex] << 24) | ((UInt32)buffer[bufferIndex + 1] << 16) | ((UInt32)buffer[bufferIndex + 2] << 8) | buffer[bufferIndex + 3];
            bufferIndex += 4;
            header.headerCrc16 = (UInt16)(((UInt16)buffer[bufferIndex] << 8) | (UInt16)buffer[bufferIndex + 1]);
            bufferIndex += 2;
            return header;
        }
        #endregion


        #region DataInterfaceUdp
        // Private members for UDP stream reception
        private int framesReceivedCounter = 0;
        private int framesReceivedCounterTemp = 0;

        private int framesDrawnCounter = 0;
        private int framesDrawnCounterTemp = 0;

        private int frameFaultCounter = 0;
        private int framesNotDrawnCounter = 0;

        private Bitmap imgDist, imgAmp;

        private Thread udpStreamReceiverThread;
        private Thread frameHandlerThread;
        private Semaphore frameQueueSem = new Semaphore(0, int.MaxValue);
        private Queue<FrameReceivedArgs> frameQueue = new Queue<FrameReceivedArgs>();
        private object controlInterfaceLock = new object();
        private bool abortUdpStreamReceiver = false;


        // This struct holds the data from the data interface header
        // The frame queue consists of intances of this struct
        // All information neccesary to draw and print the screen is included
        public struct FrameReceivedArgs
        {
            public List<List<UInt32>> data;
            public UInt16 xRes;
            public UInt16 yRes;
            public UInt16 dataFormat;
            public UInt32 timeStamp;
            public UInt16 frameCounter;
            public double mainTemp;
            public double ledTemp;
        }


        // Function runnung in a separate thread
        // It repeatedly reads UDP packets, parses the header and data and enqueues the resulting data frame into the frame queue
        private void udpStreamReceiver()
        {
            try
            {
                IPAddress addr;
                int port;
                try
                {
                    addr = IPAddress.Parse(tbUdpAddr.Text);
                    port = Convert.ToInt32(tbUdpPort.Text);
                }
                catch (Exception exc)
                {
                    MessageBox.Show("Data interface: Enter a valid IP address and port, please");
                    return;
                }
                UdpClient udpClient;
                udpClient = new UdpClient(port);
                udpClient.Client.ReceiveBufferSize = 7683000;
                udpClient.Client.ReceiveTimeout = 1000;
                try
                {
                    if (addr.ToString().Split('.')[0] == "224")
                    {
                        foreach (NetworkInterface networkInterface in NetworkInterface.GetAllNetworkInterfaces())
                        {
                            foreach (UnicastIPAddressInformation unicastAddress in networkInterface.GetIPProperties().UnicastAddresses)
                            {
                                if (unicastAddress.Address.AddressFamily == AddressFamily.InterNetwork)
                                {
                                    udpClient.JoinMulticastGroup(addr, unicastAddress.Address);
                                }
                            }
                        }
                        addr = IPAddress.Any;
                        port = 0;
                    }
                }
                catch
                {
                    MessageBox.Show("Data interface: Error with network interfaces");
                    return;
                }

                IPEndPoint ep = new IPEndPoint(addr, port);
                byte[] buffer;
                SortedList<int, byte[]> currentFrame = new SortedList<int, byte[]>();
                SortedList<int, byte[]> lastFrame = new SortedList<int, byte[]>();
                int nLastFrameSize = 0;
                int nLastFrameBytesReceived = 0;
                int nCurrentFrameSize = 0;
                int nCurrentFrameBytesReceived = 0;
                int nLastFrameCounter = int.MaxValue;
                int nCurrentFrameCounter = int.MaxValue;
                List<byte> list = new List<byte>();
                bool bParseList = false;
                CRCTool crc32 = new CRCTool();
                crc32.Init(CRCTool.CRCCode.CRC32);
                CRCTool crc16 = new CRCTool();
                crc16.Init(CRCTool.CRCCode.CRC16);

                abortUdpStreamReceiver = false;
                while (!abortUdpStreamReceiver)
                {
                    int nHeaderVersion = 0;
                    int nFrameCounter = 0;
                    int nPacketCounter = 0;
                    int nPacketSize = 0;
                    int nTotalLength = 0;
                    byte[] reserved;
                    ulong nCRC32 = 0;
                    ulong nCRC32calc = 0;
                    ulong nFlags = 0;
                    int headerLength = 0;
                    int i = 0;
                    try
                    {
                        buffer = udpClient.Receive(ref ep);
                        if (buffer.Length > 2)
                        {
                            nHeaderVersion = (int)(((ushort)buffer[i] << 8) | (ushort)buffer[i + 1]);
                            i += 2;
                            switch (nHeaderVersion)
                            {
                                case 1:
                                    {
                                        headerLength = 32;
                                        if (buffer.Length >= headerLength)
                                        {
                                            nFrameCounter = (int)(((ushort)buffer[i] << 8) | (ushort)buffer[i + 1]);
                                            i += 2;
                                            nPacketCounter = (int)(((ushort)buffer[i] << 8) | (ushort)buffer[i + 1]);
                                            i += 2;
                                            nPacketSize = (int)(((ushort)buffer[i] << 8) | (ushort)buffer[i + 1]);
                                            i += 2;
                                            nTotalLength = (int)(((ulong)buffer[i] << 24) | ((ulong)buffer[i + 1] << 16) | ((ulong)buffer[i + 2] << 8) | (ulong)buffer[i + 3]);
                                            i += 4;
                                            nCRC32 = (ulong)(((ulong)buffer[i] << 24) | ((ulong)buffer[i + 1] << 16) | ((ulong)buffer[i + 2] << 8) | (ulong)buffer[i + 3]);
                                            buffer[i] = 0;
                                            buffer[i + 1] = 0;
                                            buffer[i + 2] = 0;
                                            buffer[i + 3] = 0;
                                            i += 4;
                                            nFlags = (ulong)(((ulong)buffer[i] << 24) | ((ulong)buffer[i + 1] << 16) | ((ulong)buffer[i + 2] << 8) | (ulong)buffer[i + 3]);
                                            i += 4;
                                            reserved = new byte[12];
                                            Array.Copy(buffer, i, reserved, 0, reserved.Length);
                                            i += reserved.Length;
                                            nCRC32calc = crc32.crctablefast(buffer);
                                            if ((nCRC32 != nCRC32calc) && ((nFlags & 0x01) == 0))
                                            {
                                                continue;
                                            }
                                            else
                                            {
                                                if (nFrameCounter != nCurrentFrameCounter)
                                                {
                                                    nLastFrameCounter = nCurrentFrameCounter;
                                                    nLastFrameBytesReceived = nCurrentFrameBytesReceived;
                                                    nLastFrameSize = nCurrentFrameSize;
                                                    lastFrame = currentFrame;
                                                    currentFrame = new SortedList<int, byte[]>();
                                                    nCurrentFrameSize = nTotalLength;
                                                    nCurrentFrameCounter = nFrameCounter;
                                                    nCurrentFrameBytesReceived = 0;

                                                }
                                                else
                                                {

                                                }
                                                byte[] temp = new byte[buffer.Length - headerLength];
                                                Array.Copy(buffer, headerLength, temp, 0, temp.Length);
                                                try
                                                {
                                                    currentFrame.Add(nPacketCounter, temp);
                                                    nCurrentFrameBytesReceived += temp.Length;
                                                }
                                                catch (ArgumentNullException ex)
                                                {

                                                }
                                                catch (ArgumentException ex)
                                                {

                                                }

                                            }
                                        }
                                        break;
                                    }
                                default:
                                    {
                                        continue;
                                    }
                            }
                            if (nLastFrameSize > 0 && nLastFrameSize <= nLastFrameBytesReceived)
                            {
                                list = new List<byte>();
                                for (int j = 0; j < lastFrame.Count; j++)
                                {
                                    list.AddRange(lastFrame.Values[j]);
                                }
                                lastFrame = new SortedList<int, byte[]>();
                                bParseList = true;
                                nLastFrameBytesReceived = 0;
                                nLastFrameCounter = 0;
                                nLastFrameSize = 0;
                            }
                            else if (nCurrentFrameSize > 0 && nCurrentFrameSize <= nCurrentFrameBytesReceived)
                            {
                                list = new List<byte>();
                                for (int j = 0; j < currentFrame.Count; j++)
                                {
                                    list.AddRange(currentFrame.Values[j]);
                                }
                                currentFrame = new SortedList<int, byte[]>();
                                bParseList = true;
                                nCurrentFrameBytesReceived = 0;
                                nCurrentFrameCounter = 0;
                                nCurrentFrameSize = 0;
                            }
                            if (bParseList)
                            {
                                bParseList = false;
                                FrameReceivedArgs frame = new FrameReceivedArgs();
                                UInt16 headerVersion = 0;
                                byte bytesPerPixel = 0;
                                byte nofChannels = 0;
                                UInt16 fwVersion = 0;
                                framesReceivedCounter++;
                                if (frameQueue.Count <= 5)
                                {
                                    if (parseHeader(list.ToArray(), 2, out headerVersion, out frame.xRes, out frame.yRes, out bytesPerPixel, out nofChannels, out frame.dataFormat, out frame.timeStamp, out frame.frameCounter, out frame.mainTemp, out frame.ledTemp, out fwVersion) > 0)
                                    {
                                        parseData(list.ToArray(), CONTROL_IF_HEADER_SIZE, out frame.data, nofChannels, frame.dataFormat, bytesPerPixel, frame.xRes, frame.yRes);
                                        frameQueue.Enqueue(frame);
                                        frameQueueSem.Release();
                                        // Statistics
                                        int pixel_index = mouse_pos_x + mouse_pos_y * frame.xRes;
                                        if (pixel_index > 0 && pixel_index < frame.xRes * frame.yRes)
                                        {
                                            if (frame.dataFormat >> 3 == 0 || frame.dataFormat >> 3 == 3 || frame.dataFormat >> 3 == 4)
                                            {
                                                // channel 1 is Dist
                                                statistics_ch1.AddElement(frame.data[0], frame.xRes, frame.yRes);
                                                if (statistics_ch1.FillSize == statistics_ch1.MaxSize)
                                                {
                                                    List<double> avgs = statistics_ch1.GetAverage();
                                                    if (avgs.Count > pixel_index)
                                                    {
                                                        string msg_statistics_ch1 = "Avg: " + avgs[pixel_index].ToString("F2");
                                                        List<double> devs = statistics_ch1.GetStdDeviation(avgs);
                                                        if (devs.Count > pixel_index)
                                                        {
                                                            msg_statistics_ch1 += "  Stddev: " + devs[pixel_index].ToString("F2");
                                                        }
                                                        statistics_ch1.Reset();
                                                        Invoke(new MethodInvoker(delegate()
                                                        {
                                                            lbStatisticsCh1.Text = msg_statistics_ch1;
                                                        }));
                                                    }
                                                }
                                            }
                                            if (frame.dataFormat >> 3 == 0 || frame.dataFormat >> 3 == 3 || frame.dataFormat >> 3 == 4)
                                            {
                                                // channel 2 is Amp
                                                int channelIdx = 3;
                                                if (frame.dataFormat >> 3 == 0)
                                                {
                                                    channelIdx = 1;
                                                }
                                                statistics_ch2.AddElement(frame.data[channelIdx], frame.xRes, frame.yRes);
                                                if (statistics_ch2.FillSize == statistics_ch2.MaxSize)
                                                {
                                                    List<double> avgs = statistics_ch2.GetAverage();
                                                    if (avgs.Count > pixel_index)
                                                    {
                                                        string msg_statistics_ch2 = "Avg: " + avgs[pixel_index].ToString("F2");
                                                        List<double> devs = statistics_ch2.GetStdDeviation(avgs);
                                                        if (devs.Count > pixel_index)
                                                        {
                                                            msg_statistics_ch2 += "  Stddev: " + devs[pixel_index].ToString("F2");
                                                        }
                                                        statistics_ch2.Reset();
                                                        Invoke(new MethodInvoker(delegate()
                                                        {
                                                            lbStatisticsCh2.Text = msg_statistics_ch2;
                                                        }));
                                                    }
                                                }
                                            }
                                        }
                                    }
                                    else
                                    {
                                        frameFaultCounter++;
                                    }
                                }
                                else
                                {
                                    framesNotDrawnCounter++;
                                }
                            }
                        }
                        else
                        {
                            Thread.Sleep(1000);
                        }
                    }
                    catch (ObjectDisposedException ex)
                    {
                    }
                    catch (SocketException ex)
                    {
                    }
                }
                udpClient.Client.Close();
                udpClient.Close();
            }
            catch (Exception exc)
            {
                MessageBox.Show("Data interface: Unable to connect");
            }
        }


        // This function parses the header of the frame received from the data interface
        private int parseHeader(byte[] data, int offset,
            out UInt16 headerVersion, out UInt16 xRes, out UInt16 yRes, out byte bytesPerPixel, out byte nofChannels,
            out UInt16 dataFormat, out UInt32 timeStamp, out UInt16 frameCounter,
            out double mainTemp, out double ledTemp, out UInt16 fwVersion)
        {
            int i = offset;
            xRes = yRes = dataFormat = frameCounter = fwVersion = 0;
            timeStamp = 0;
            nofChannels = bytesPerPixel = 0;
            mainTemp = ledTemp = 0;

            headerVersion = (UInt16)((UInt16)(((UInt16)data[i]) << 8) | ((UInt16)data[i + 1]));
            i += 2;
            if (headerVersion != 3)
            {
                return 0;
            }
            CRCTool crcTool = new CRCTool();
            crcTool.Init(CRCTool.CRCCode.CRC16);
            UInt16 crc16 = (UInt16)((UInt16)(((UInt16)data[62]) << 8) | ((UInt16)data[63]));
            UInt16 crc16Calc = crcTool.crctable16(data, 2, 60);
            if (crc16 != crc16Calc || crc16 == 0xffff)
            {
                return 0;
            }
            xRes = (UInt16)((UInt16)(((UInt16)data[i]) << 8) | ((UInt16)data[i + 1]));
            i += 2;
            yRes = (UInt16)((UInt16)(((UInt16)data[i]) << 8) | ((UInt16)data[i + 1]));
            i += 2;
            nofChannels = data[i++];
            bytesPerPixel = data[i++];
            dataFormat = (UInt16)((UInt16)(((UInt16)data[i]) << 8) | ((UInt16)data[i + 1]));
            i += 2;
            timeStamp = (UInt32)((UInt32)(((UInt32)data[i]) << 24) | (((UInt32)data[i + 1]) << 16) | (((UInt32)data[i + 2]) << 8) | ((UInt32)data[i + 3]));
            i += 4;
            frameCounter = (UInt16)((UInt16)(((UInt16)data[i]) << 8) | ((UInt16)data[i + 1]));
            i += 10;
            mainTemp = data[i++] - 50;
            ledTemp = data[i++] - 50;
            fwVersion = (UInt16)((UInt16)(((UInt16)data[i]) << 8) | ((UInt16)data[i + 1]));
            i += 8;
            return CONTROL_IF_HEADER_SIZE;
        }


        // This function parses the data block of the frame received from the data interface
        private void parseData(byte[] temp, int offset, out List<List<UInt32>> data, int nofChannels, int dataFormat, int bytesPerPixel, int xRes, int yRes)
        {
            data = new List<List<UInt32>>();
            for (int i = 0; i < nofChannels; i++)
            {
                data.Add(new List<UInt32>());
            }
            if (bytesPerPixel > 4)
            {
                throw new Exception("Only 4 bytes per pixel supported");
            }
            int index = offset;
            for (int y = 0; y < nofChannels; y++)
            {
                for (int x = 0; x < (xRes * yRes); x++)
                {
                    UInt32 sample = 0;
                    for (int j = 0; j < bytesPerPixel; j++)
                    {
                        sample |= ((UInt32)temp[index + j]) << (j * 8);
                    }
                    index += bytesPerPixel;
                    data[y].Add(sample);
                }
            }
        }


        // This function runs in a seperate thread
        // It waits for data from the frame queue and transforms it into two bitmaps which are then displayed in the
        // form window along with some information in text form
        private void frameHandler()
        {
            while (true)
            {
                frameQueueSem.WaitOne();
                if (frameQueue.Count > 0)
                {
                    string msg_curr_val_ch1 = "Dist: n.a. 0/" + statistics_ch1.MaxSize;
                    string msg_curr_val_ch2 = "Amp: n.a. 0/" + statistics_ch2.MaxSize;
                    FrameReceivedArgs frame = frameQueue.Dequeue();
                    if (frame.xRes <= 0 || frame.yRes <= 0)
                    {
                        continue;
                    }
                    if (imgDist == null)
                    {
                        imgDist = new Bitmap(frame.xRes, frame.yRes, System.Drawing.Imaging.PixelFormat.Format24bppRgb);
                    }
                    if (imgAmp == null)
                    {
                        imgAmp = new Bitmap(frame.xRes, frame.yRes, System.Drawing.Imaging.PixelFormat.Format24bppRgb);
                    }
                    List<byte> img_dist_rgb = new List<byte>();
                    List<byte> img_amp_rgb = new List<byte>();
                    if (frame.dataFormat >> 3 == 0 || frame.dataFormat >> 3 == 3 || frame.dataFormat >> 3 == 4)
                    {
                        if (mouse_pos_x > 0 && mouse_pos_y > 0 && mouse_pos_x < frame.xRes && mouse_pos_y < frame.yRes)
                        {
                            msg_curr_val_ch1 = "Dist: " + frame.data[0][mouse_pos_x + mouse_pos_y * frame.xRes].ToString("F2") + "  " + statistics_ch1.FillSize + "/" + statistics_ch1.MaxSize; ;
                        }
                        int offsetDist, scaleDist;
                        try
                        {
                            offsetDist = Convert.ToInt32(tbChannel1RangeMin.Text);
                            Invoke(new MethodInvoker(delegate()
                            {
                                tbChannel1RangeMin.BackColor = Color.White;
                            }));
                        }
                        catch (FormatException exc)
                        {
                            offsetDist = 0;
                            Invoke(new MethodInvoker(delegate()
                            {
                                tbChannel1RangeMin.BackColor = Color.Orange;
                            }));
                        }
                        if (offsetDist < 0)
                        {
                            offsetDist = 0;
                            Invoke(new MethodInvoker(delegate()
                            {
                                tbChannel1RangeMin.BackColor = Color.Orange;
                            }));
                        }
                        try
                        {
                            scaleDist = Convert.ToInt32(tbChannel1RangeMax.Text) - offsetDist;
                            Invoke(new MethodInvoker(delegate()
                            {
                                tbChannel1RangeMax.BackColor = Color.White;
                            }));
                        }
                        catch (FormatException exc)
                        {
                            scaleDist = 3500;
                            Invoke(new MethodInvoker(delegate()
                            {
                                tbChannel1RangeMax.BackColor = Color.Orange;
                            }));
                        }
                        if (scaleDist <= 0)
                        {
                            scaleDist = 3500;
                            Invoke(new MethodInvoker(delegate()
                            {
                                tbChannel1RangeMax.BackColor = Color.Orange;
                            }));
                        }
                        int index = 0;
                        for (int y = 0; y < frame.yRes; y++)
                        {
                            for (int x = 0; x < frame.xRes; x++)
                            {
                                byte r, g, b;
                                if (frame.data[0][index] == 0)
                                {
                                    r = byte.MaxValue;
                                    g = byte.MaxValue;
                                    b = byte.MaxValue;
                                }
                                else if (frame.data[0][index] == 0xffff)
                                {
                                    r = 0;
                                    g = 0;
                                    b = 0;
                                }
                                else
                                {
                                    long pix = (frame.data[0][index] - offsetDist) * UInt16.MaxValue / scaleDist;
                                    if (pix < UInt16.MinValue)
                                    {
                                        pix = UInt16.MinValue;
                                    }
                                    else if (pix > UInt16.MaxValue)
                                    {
                                        pix = UInt16.MaxValue;
                                    }

                                    if (pix <= UInt16.MaxValue / 4)
                                    {
                                        r = byte.MaxValue;
                                        g = (byte)(pix * byte.MaxValue / (UInt16.MaxValue / 4));
                                        b = 0;
                                    }
                                    else
                                    {
                                        pix -= UInt16.MaxValue / 4;
                                        if (pix <= UInt16.MaxValue / 4)
                                        {
                                            r = (byte)(byte.MaxValue - pix * byte.MaxValue / (UInt16.MaxValue / 4));
                                            g = byte.MaxValue;
                                            b = 0;
                                        }
                                        else
                                        {
                                            pix -= UInt16.MaxValue / 4;
                                            if (pix <= UInt16.MaxValue / 4)
                                            {
                                                r = 0;
                                                g = byte.MaxValue;
                                                b = (byte)(pix * byte.MaxValue / (UInt16.MaxValue / 4));
                                            }
                                            else
                                            {
                                                pix -= UInt16.MaxValue / 4;
                                                r = 0;
                                                g = (byte)(byte.MaxValue - pix * byte.MaxValue / (UInt16.MaxValue / 4));
                                                b = byte.MaxValue;
                                            }
                                        }
                                    }
                                }
                                img_dist_rgb.Add(b);
                                img_dist_rgb.Add(g);
                                img_dist_rgb.Add(r);
                                index++;
                            }
                        }
                    }
                    if (frame.dataFormat >> 3 == 0 || frame.dataFormat >> 3 == 4)
                    {
                        int channelIdx = 3;
                        if (frame.dataFormat >> 3 == 0)
                        {
                            channelIdx = 1;
                        }
                        if (mouse_pos_x > 0 && mouse_pos_y > 0 && mouse_pos_x < frame.xRes && mouse_pos_y < frame.yRes)
                        {
                            msg_curr_val_ch2 = "Amp: " + frame.data[channelIdx][mouse_pos_x + mouse_pos_y * frame.xRes].ToString() + "  " + statistics_ch2.FillSize + "/" + statistics_ch2.MaxSize;
                        }
                        int offsetAmp, scaleAmp;
                        try
                        {
                            offsetAmp = Convert.ToInt32(tbChannel2RangeMin.Text);
                            Invoke(new MethodInvoker(delegate()
                            {
                                tbChannel2RangeMin.BackColor = Color.White;
                            }));
                        }
                        catch (FormatException exc)
                        {
                            offsetAmp = 0;
                            Invoke(new MethodInvoker(delegate()
                            {
                                tbChannel2RangeMin.BackColor = Color.Orange;
                            }));
                        }
                        if (offsetAmp < 0)
                        {
                            offsetAmp = 0;
                            Invoke(new MethodInvoker(delegate()
                            {
                                tbChannel2RangeMin.BackColor = Color.Orange;
                            }));
                        }
                        try
                        {
                            scaleAmp = Convert.ToInt32(tbChannel2RangeMax.Text) - offsetAmp;
                            Invoke(new MethodInvoker(delegate()
                            {
                                tbChannel2RangeMax.BackColor = Color.White;
                            }));
                        }
                        catch (FormatException exc)
                        {
                            scaleAmp = 2500;
                            Invoke(new MethodInvoker(delegate()
                            {
                                tbChannel2RangeMax.BackColor = Color.Orange;
                            }));
                        }
                        if (scaleAmp <= 0)
                        {
                            scaleAmp = 2500;
                            Invoke(new MethodInvoker(delegate()
                            {
                                tbChannel2RangeMax.BackColor = Color.Orange;
                            }));
                        }
                        int index = 0;
                        for (int y = 0; y < frame.yRes; y++)
                        {
                            for (int x = 0; x < frame.xRes; x++)
                            {
                                long ampScaled = (frame.data[channelIdx][index] - offsetAmp) * UInt16.MaxValue / scaleAmp;
                                byte brightness;
                                if (ampScaled >= UInt16.MaxValue)
                                {
                                    brightness = byte.MaxValue;
                                }
                                else if (ampScaled <= UInt16.MinValue)
                                {
                                    brightness = byte.MinValue;
                                }
                                else
                                {
                                    brightness = (byte)(ampScaled >> 8);
                                }
                                img_amp_rgb.Add(brightness);
                                img_amp_rgb.Add(brightness);
                                img_amp_rgb.Add(brightness);
                                index++;
                            }
                        }
                    }

                    try
                    {
                        // copy first bitmap data
                        System.Drawing.Imaging.BitmapData bmpData = imgDist.LockBits(new Rectangle(0, 0, imgDist.Width, imgDist.Height), System.Drawing.Imaging.ImageLockMode.ReadWrite, imgDist.PixelFormat);
                        IntPtr ptr = bmpData.Scan0;
                        System.Runtime.InteropServices.Marshal.Copy(img_dist_rgb.ToArray(), 0, bmpData.Scan0, img_dist_rgb.Count);
                        imgDist.UnlockBits(bmpData);

                        // copy second bitmap data
                        bmpData = imgAmp.LockBits(new Rectangle(0, 0, imgAmp.Width, imgAmp.Height), System.Drawing.Imaging.ImageLockMode.ReadWrite, imgAmp.PixelFormat);
                        ptr = bmpData.Scan0;
                        System.Runtime.InteropServices.Marshal.Copy(img_amp_rgb.ToArray(), 0, bmpData.Scan0, img_amp_rgb.Count);
                        imgAmp.UnlockBits(bmpData);

                        framesDrawnCounter++;
                        Invoke(new publishDelegate(publish), new object[] { imgDist, imgAmp, frame.dataFormat, frame.timeStamp, frame.frameCounter, Convert.ToDouble(frame.mainTemp), Convert.ToDouble(frame.ledTemp), msg_curr_val_ch1, msg_curr_val_ch2 });
                    }
                    catch (Exception exc) {
                        imgAmp = null;
                        imgDist = null;
                    }
                }
            }
        }



        // This function is invoked, in order to be able to access the visual elements
        // The information in the function's arguments are the displayed in the main form
        private delegate void publishDelegate(Bitmap image_ch1, Bitmap image_ch2, UInt16 data_format, UInt32 timestamp, UInt16 frame_counter, double main_temp, double led_temp, string msg_curr_val_ch1, string msg_curr_val_ch2);
        private void publish(Bitmap image_ch1, Bitmap image_ch2, UInt16 data_format, UInt32 timestamp, UInt16 frame_counter, double main_temp, double led_temp, string msg_curr_val_ch1, string msg_curr_val_ch2)
        {
            pbImageCh1.Image = new Bitmap(image_ch1);
            pbImageCh2.Image = new Bitmap(image_ch2);
            string text = "";
            text += "Frame: " + frame_counter;
            text = text.PadRight(20);
            text += "Main: " + main_temp + " °C";
            text = text.PadRight(40);
            text += "Led: " + led_temp + " °C";
            text = text.PadRight(60);
            text += "Faulty frames: " + frameFaultCounter;
            lblFrameCounter.Text = text;
            if (data_format >> 3 == 0)
            {
                lbChannel1Name.Text = "Dist";
                lbChannel2Name.Text = "Amp";
            }
            else if (data_format >> 3 == 3)
            {
                lbChannel1Name.Text = "X dist";
                lbChannel2Name.Text = "";
            }
            else if (data_format >> 3 == 4)
            {
                lbChannel1Name.Text = "X dist";
                lbChannel2Name.Text = "Amp";
            }
            else
            {
                lbChannel1Name.Text = "N.a.";
                lbChannel2Name.Text = "N.a.";
            }
            if (mouse_pos_x > 0 && mouse_pos_y > 0 && mouse_pos_x < image_ch1.Width && mouse_pos_y < image_ch1.Height)
            {
                lbStatisticsPos.Text = "(x, y): (" + mouse_pos_x + ", " + mouse_pos_y + ")";
            }
            else
            {
                lbStatisticsPos.Text = "";
            }
            lbCurrValCh1.Text = msg_curr_val_ch1;
            lbCurrValCh2.Text = msg_curr_val_ch2;
        }
        #endregion


        #region ClickEventHandler
        private void bttnTintSet_Click(object sender, EventArgs e)
        {
            UInt16 valueToWrite;
            try
            {
                valueToWrite = Convert.ToUInt16(tbTint.Text);
            }
            catch (Exception exc)
            {
                MessageBox.Show("Enter a valid integration time, please");
                return;
            }

            byte status = writeRegister(REG_INTEGRATION_TIME, valueToWrite);
            if (status != 0)
            {
                tbTint.BackColor = Color.Red;
            }
            else
            {
                tbTint.BackColor = Color.Green;
            }
            timerTintColor.Enabled = false;
            timerTintColor.Interval = 1000;
            timerTintColor.Enabled = true;
        }


        private void bttnTintGet_Click(object sender, EventArgs e)
        {
            UInt16 regValue;
            byte status = readRegister(REG_INTEGRATION_TIME, out regValue);
            if (status != 0)
            {
                tbTint.BackColor = Color.Red;
            }
            else
            {
                tbTint.BackColor = Color.Green;
                tbTint.Text = Convert.ToString(regValue);
            }
            timerTintColor.Enabled = false;
            timerTintColor.Interval = 1000;
            timerTintColor.Enabled = true;
        }


        private void bttnFrameRateSet_Click(object sender, EventArgs e)
        {
            UInt16 valueToWrite;
            try
            {
                valueToWrite = Convert.ToUInt16(tbFrameRate.Text);
            }
            catch (Exception exc)
            {
                MessageBox.Show("Enter a valid integration time, please");
                return;
            }

            byte status = writeRegister(REG_FRAME_RATE, valueToWrite);
            if (status != 0)
            {
                tbFrameRate.BackColor = Color.Red;
            }
            else
            {
                tbFrameRate.BackColor = Color.Green;
            }
            timerFrameRateColor.Enabled = false;
            timerFrameRateColor.Interval = 1000;
            timerFrameRateColor.Enabled = true;
        }


        private void bttnFrameRateGet_Click(object sender, EventArgs e)
        {
            UInt16 regValue;
            byte status = readRegister(REG_FRAME_RATE, out regValue);
            if (status != 0)
            {
                tbFrameRate.BackColor = Color.Red;
            }
            else
            {
                tbFrameRate.BackColor = Color.Green;
                tbFrameRate.Text = Convert.ToString(regValue);
            }
            timerFrameRateColor.Enabled = false;
            timerFrameRateColor.Interval = 1000;
            timerFrameRateColor.Enabled = true;
        }


        private void bttnCustomRegSet_Click(object sender, EventArgs e)
        {
            UInt16 addrToWrite;
            try
            {
                addrToWrite = Convert.ToUInt16(tbCustomRegAddr.Text, (int)16);
            }
            catch (Exception exc)
            {
                MessageBox.Show("Enter a valid register address, please");
                return;
            }

            UInt16 valueToWrite;
            try
            {
                valueToWrite = Convert.ToUInt16(tbCustomRegValue.Text);
            }
            catch (Exception exc)
            {
                MessageBox.Show("Enter a valid register value, please");
                return;
            }

            byte status = writeRegister(addrToWrite, valueToWrite);
            if (status != 0)
            {
                tbCustomRegValue.BackColor = Color.Red;
            }
            else
            {
                tbCustomRegValue.BackColor = Color.Green;
            }
            timerCustomRegColor.Enabled = false;
            timerCustomRegColor.Interval = 1000;
            timerCustomRegColor.Enabled = true;
        }


        private void bttnCustomRegGet_Click(object sender, EventArgs e)
        {
            UInt16 addrToRead;
            try
            {
                addrToRead = Convert.ToUInt16(tbCustomRegAddr.Text, (int)16);
            }
            catch (Exception exc)
            {
                MessageBox.Show("Enter a valid register address, please");
                return;
            }

            UInt16 regValue;
            byte status = readRegister(addrToRead, out regValue);
            if (status != 0)
            {
                tbCustomRegValue.BackColor = Color.Red;
            }
            else
            {
                tbCustomRegValue.BackColor = Color.Green;
                tbCustomRegValue.Text = Convert.ToString(regValue);
            }
            timerCustomRegColor.Enabled = false;
            timerCustomRegColor.Interval = 1000;
            timerCustomRegColor.Enabled = true;
        }


        private void bttnUdpApply_Click(object sender, EventArgs e)
        {
            abortUdpStreamReceiver = true;
            while (udpStreamReceiverThread.IsAlive) ;
            udpStreamReceiverThread = new Thread(udpStreamReceiver);
            udpStreamReceiverThread.Name = "UDPstreamReceiver";
            udpStreamReceiverThread.Start();
        }

        private void cbImageZoom_CheckedChanged(object sender, EventArgs e)
        {
            if (cbImageZoom.Checked)
            {
                pbImageCh1.SizeMode = PictureBoxSizeMode.Zoom;
                pbImageCh2.SizeMode = PictureBoxSizeMode.Zoom;
            }
            else
            {
                pbImageCh1.SizeMode = PictureBoxSizeMode.Normal;
                pbImageCh2.SizeMode = PictureBoxSizeMode.Normal;
            }
        }
        #endregion


        #region GeneralEventHandler
        private int mouse_pos_x = -1;
        private int mouse_pos_y = -1;
        Statistics statistics_ch1 = new Statistics(1);
        Statistics statistics_ch2 = new Statistics(1);
        private void SentisTofVisualizerForm_Resize(object sender, EventArgs e)
        {
            pbImageCh1.Width = this.Size.Width / 2 - 15;
            pbImageCh1.Height = pbImageCh1.Width * 3 / 4;
            pbImageCh1.Left = 5;
            pbImageCh2.Width = this.Size.Width / 2 - 15;
            pbImageCh2.Height = pbImageCh2.Width * 3 / 4;
            pbImageCh2.Left = this.Size.Width / 2 - 5;
        }


        private void SentisTofVisualizerForm_FormClosing(object sender, FormClosingEventArgs e)
        {
            frameHandlerThread.Abort();
            abortUdpStreamReceiver = true;
            while (udpStreamReceiverThread.IsAlive) ;
        }


        private void timerTintColor_Tick(object sender, EventArgs e)
        {
            tbTint.BackColor = Color.White;
            timerTintColor.Enabled = false;
        }


        private void timerFrameRate_Tick(object sender, EventArgs e)
        {
            int frameRateDrawn = framesDrawnCounter - framesDrawnCounterTemp;
            framesDrawnCounterTemp = framesDrawnCounter;
            int frameRateReceived = framesReceivedCounter - framesReceivedCounterTemp;
            framesReceivedCounterTemp = framesReceivedCounter;
            Invoke(new MethodInvoker(delegate()
            {
                string text = "Fps drawn: " + frameRateDrawn;
                text = text.PadRight(20);
                text += "Fps received: " + frameRateReceived;
                lbFramerate.Text = text;
            }));
        }


        private void timerFrameRateColor_Tick(object sender, EventArgs e)
        {
            tbFrameRate.BackColor = Color.White;
            timerFrameRateColor.Enabled = false;
        }

        private void timerCustomRegColor_Tick(object sender, EventArgs e)
        {
            tbCustomRegValue.BackColor = Color.White;
            timerCustomRegColor.Enabled = false;
        }

        private void pb_MouseLeave(object sender, EventArgs e)
        {
            mouse_pos_x = -1;
            mouse_pos_y = -1;
        }

        private void pb_MouseMove(object sender, MouseEventArgs e)
        {
            int mouse_pos_x_before = mouse_pos_x;
            int mouse_pos_y_before = mouse_pos_y;
            try
            {
                PictureBox pb = (PictureBox)sender;
                if (pb.Image == null)
                {
                    return;
                }
                if (!cbImageZoom.Checked)
                {
                    //Point point = pb.PointToClient(new Point(e.X, e.Y));
                    mouse_pos_x = e.X;
                    mouse_pos_y = e.Y;
                    if (mouse_pos_x < 0 || mouse_pos_x >= pb.Image.Width || mouse_pos_y < 0 || mouse_pos_y >= pb.Image.Height)
                    {
                        mouse_pos_x = -1;
                        mouse_pos_y = -1;
                    }
                    return;
                }
                double scale = (double)pb.Image.Width / (double)pb.Image.Height;
                int width = 0;
                int height = 0;
                if (((double)pb.Height * scale) > (double)pb.Width)
                {
                    width = pb.Width;
                    height = (int)((double)pb.Width / scale);
                }
                else
                {
                    height = pb.Height;
                    width = (int)((double)pb.Height * scale);
                }

                double x = (double)e.X - (((double)(pb.Width - width) / 2.0));
                double y = (double)e.Y - (((double)(pb.Height - height) / 2.0));

                scale = (double)pb.Image.Width / (double)width;
                mouse_pos_x = (int)(x * scale) + 1;
                mouse_pos_y = (int)(y * scale) + 1;
            }
            catch (Exception exc)
            {
                mouse_pos_x = -1;
                mouse_pos_y = -1;
            }
            if (mouse_pos_x != mouse_pos_x_before || mouse_pos_y != mouse_pos_y_before)
            {
                statistics_ch1.Reset();
                statistics_ch2.Reset();
                Invoke(new MethodInvoker(delegate()
                {
                    lbStatisticsCh1.Text = "Avg: n.a.  Stddev: n.a.";
                    lbStatisticsCh2.Text = "Avg: n.a.  Stddev: n.a.";
                }));
            }
        }

        private void txtStatsFrameCount_TextChanged(object sender, EventArgs e)
        {
            int frame_count = 1;
            try
            {
                frame_count = Convert.ToInt32(txtStatsFrameCount.Text);
                txtStatsFrameCount.BackColor = Color.White;
            }
            catch
            {
                txtStatsFrameCount.BackColor = Color.Orange;
            }
            if (frame_count < 1)
            {
                frame_count = 1;
                txtStatsFrameCount.BackColor = Color.Orange;
            }
            statistics_ch1.MaxSize = frame_count;
            statistics_ch2.MaxSize = frame_count;
        }
        #endregion

        private void SentisTofVisualizerForm_Load(object sender, EventArgs e)
        {
            Location = Properties.Settings.Default.main_pos;
            Size = Properties.Settings.Default.main_size;
        }

        private void SentisTofVisualizerForm_ResizeEnd(object sender, EventArgs e)
        {
            Properties.Settings.Default.main_pos = Location;
            Properties.Settings.Default.main_size = Size;
            Properties.Settings.Default.Save();
        }

        private void tbTcpAddr_TextChanged(object sender, EventArgs e)
        {
            Properties.Settings.Default.tcp_addr = tbTcpAddr.Text;
            Properties.Settings.Default.Save();
        }

        private void tbTcpPort_TextChanged(object sender, EventArgs e)
        {
            Properties.Settings.Default.tcp_port = tbTcpPort.Text;
            Properties.Settings.Default.Save();
        }

        private void tbUdpAddr_TextChanged(object sender, EventArgs e)
        {
            Properties.Settings.Default.udp_addr = tbUdpAddr.Text;
            Properties.Settings.Default.Save();
        }

        private void tbUdpPort_TextChanged(object sender, EventArgs e)
        {
            Properties.Settings.Default.udp_port = tbUdpPort.Text;
            Properties.Settings.Default.Save();
        }
    }
}
